
package com.paydevice.smartpos.demo.smartcard;

import android.support.v7.app.AppCompatActivity;
import android.app.AlertDialog;
import android.app.AlertDialog.Builder;
import android.content.Context;
import android.content.DialogInterface;
import android.os.AsyncTask;
import android.os.Bundle;
import android.os.Handler;
import android.os.Message;
import android.util.Log;
import android.view.View;
import android.view.View.OnClickListener;
import android.widget.Button;
import android.widget.TextView;

import com.paydevice.smartpos.demo.R;
import com.paydevice.smartpos.demo.MyToast;
import com.paydevice.smartpos.demo.Utils;
import com.paydevice.smartpos.sdk.SmartPosException;
import com.paydevice.smartpos.sdk.smartcard.SmartCardManager;

public class ISO7816Activity extends AppCompatActivity {

    private static final String TAG = "ISO7816Activity";

    private static final String KEY_RESULT = "result";
    private static final String KEY_SN = "sn";
    private static final String KEY_PROTOCOL = "protocol";
    private static final String KEY_ERR_CODE = "code";

    private static final int MSG_OPEN_RESULT = 1;
    private static final int MSG_CLOSE_RESULT = 2;
    private boolean mOpenFlag = false;
    
    private SmartCardManager mSmartCardManager;

	private MyToast mToast;
    private TextView mText;
    private TextView mEditText;
    private Button mOpenBtn;
    private Button mCloseBtn;
    private Button mAtrBtn;
    private Button mApduBtn;
    private OpenTask mOpenTask;
    private CloseTask mCloseTask;

    private Builder mSlotDialog;
    private int mSlotSelected = 0;
    private int mSlotCount = 1;

    public Handler mHandler = new Handler() {
        public void handleMessage(Message msg) {
            Boolean result = msg.getData().getBoolean(KEY_RESULT);
            switch (msg.what) {
                case MSG_CLOSE_RESULT:
                    if (result) {
                        mToast.showToast(R.string.smartcard_close_success);
                    } else {
						int errCode = msg.getData().getInt(KEY_ERR_CODE);
                        mCloseBtn.setEnabled(false);
						mText.setText("errCode: " + Integer.toHexString(errCode) + " " + SmartCardManager.errCode2String(errCode));
                        mToast.showToast(R.string.smartcard_close_failed);
                    }
                    mOpenBtn.setEnabled(true);
                    mAtrBtn.setEnabled(false);
                    mApduBtn.setEnabled(false);
                    mOpenFlag = false;
                    break;


                case MSG_OPEN_RESULT:
                    if (result) {
                        String sn = msg.getData().getString(KEY_SN);
                        int protocol = msg.getData().getInt(KEY_PROTOCOL);
                        Log.d(TAG, "SN:" + sn + " Protocol:" + ((protocol == SmartCardManager.PROTOCOL_T0) ? "T0" : "T1"));
                        mText.setText("SN:" + sn + " Protocol:" + ((protocol == SmartCardManager.PROTOCOL_T0) ? "T0" : "T1"));
                        mCloseBtn.setEnabled(true);
                        mAtrBtn.setEnabled(true);
                        mApduBtn.setEnabled(true);
                        mOpenFlag = true;
                        mToast.showToast(R.string.smartcard_open_success);
                    } else {
						int errCode = msg.getData().getInt(KEY_ERR_CODE);
                        if (errCode == SmartCardManager.SCR_ERR_NO_INIT) {
                            init();
                            if (mSlotCount > 1) {
                                mSlotDialog.show();
                            } else {
                                open();
                            }
                        } else {
                            mOpenBtn.setEnabled(true);
                            mAtrBtn.setEnabled(false);
                            mApduBtn.setEnabled(false);
							mText.setText("errCode: " + Integer.toHexString(errCode) + " " + SmartCardManager.errCode2String(errCode));
                            mToast.showToast(R.string.smartcard_open_failed);
                        }
                    }
                    break;
            }
        }
    };

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.iso7816);
        setTitle(R.string.smartcard_mode_iso7816);

		mSmartCardManager = new SmartCardManager();

		mToast = new MyToast(this);
        mText = (TextView) this.findViewById(R.id.text);
        mEditText = (TextView) this.findViewById(R.id.edit);
        mEditText.setText("0084000004");

        mOpenBtn = (Button) this.findViewById(R.id.open_btn);
        mOpenBtn.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View arg) {
                if (mSlotCount > 1) {
                    mSlotDialog.show();
                } else {
                    open();
                }
            }
        });

        mCloseBtn = (Button) this.findViewById(R.id.close_btn);
        mCloseBtn.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View arg) {
                mText.setText("");
                close();
            }
        });

        mAtrBtn = (Button) this.findViewById(R.id.atr_btn);
        mAtrBtn.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View arg) {
                getAtr();
            }
        });
        mApduBtn = (Button) this.findViewById(R.id.apdu_btn);
        mApduBtn.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View arg) {
                sendApdu();
            }
        });

        mCloseBtn.setEnabled(false);
        mAtrBtn.setEnabled(false);
        mApduBtn.setEnabled(false);

        setReaderSlotView();
		//we just init reader to get slot count, unnecessary for single slot.
        init();
        mSmartCardManager.deinit();
    }

    @Override
    protected void onPause() {
        super.onPause();
		mToast.cancel();
    }

    @Override
    public void onDestroy() {
        //NOTE: power off card and deinit
        Log.d(TAG, "onDestroy()");
        if (mOpenFlag) {
            close();
        } else {
            mSmartCardManager.deinit();
            Log.d(TAG, "Deinit");
            if (mCloseTask != null && mCloseTask.getStatus() != AsyncTask.Status.FINISHED) {
                mCloseTask.cancel(true);
            }
        }
        if (mOpenTask != null && mOpenTask.getStatus() != AsyncTask.Status.FINISHED) {
            mOpenTask.cancel(true);
        }

        super.onDestroy();
    }

    private void init() {
        int errCode;
        String errMsg = null;
        try {
			//init ISO7816 mode
            mSlotCount = mSmartCardManager.init(getApplicationContext());
            Log.d(TAG, "got slot count:" + mSlotCount);
        } catch (SmartPosException e) {
            mOpenBtn.setEnabled(false);
            errCode = e.getErrorCode();
            switch (errCode) {
                case SmartCardManager.SCR_ERR_INIT:
                    errMsg = "Reader init error!";
                    break;
            }
            mText.setText("errCode: " + Integer.toHexString(errCode) + "  " + errMsg);
            mToast.showToast(R.string.smartcard_init_err);
        }
    }

    /**
     * @return
     * @brief init smartcard reader and power on card
     */
    private void open() {
        Log.d(TAG, "open slot:" + mSlotSelected);
        mOpenBtn.setEnabled(false);
        mOpenTask = new OpenTask();
        mOpenTask.execute(this);
    }

    /**
     * @return
     * @brief deinit smartcard reader and power off card
     */
    private void close() {
        mCloseBtn.setEnabled(false);
        mCloseTask = new CloseTask();
        mCloseTask.execute();
    }

    /**
     * @return
     * @brief send APDU command
     */
    private void sendApdu() {
        try {
            Log.d(TAG, "Send APDU");
            String cmd = mEditText.getText().toString();
            byte[] apduSend = Utils.stringToBytes(cmd);
            byte[] respone = new byte[2048];//for long respone,max 65535
			int[] responeLenArray = new int[8];
            int len = 0;
			mSmartCardManager.sendAPDU(apduSend, respone, responeLenArray);
			for(int i=0;i<responeLenArray.length;i++) {
                len += responeLenArray[i];
            }
            mText.setText("APDU receive("+len+"):");
			for(int i=0;i<responeLenArray.length;i++) {
                if (responeLenArray[i] > 0) {
                    mText.append(Utils.printBytes(respone, 0, responeLenArray[i]));
                }
			}
        } catch (SmartPosException e) {
			mText.setText("errCode: " + Integer.toHexString(e.getErrorCode()) + " " + SmartCardManager.errCode2String(e.getErrorCode()));
            mToast.showToast(R.string.smartcard_send_apdu_failed);
        }
    }

    /**
     * @return
     * @brief get ATR
     */
    private void getAtr() {
        try {
            Log.d(TAG, "get ATR");
            byte[] sn = mSmartCardManager.getATR();
            mText.setText("ATR:" + Utils.bytesToHexString(sn));
        } catch (SmartPosException e) {
            mToast.showToast(R.string.smartcard_get_atr_failed);
            mText.setText("");
        }
    }

    private void setReaderSlotView() {
        final String[] arraySlot = new String[]{"slot:0", "Slot:1"};
        mSlotDialog = new AlertDialog.Builder(this);
        DialogInterface.OnClickListener Select = new DialogInterface.OnClickListener() {

            @Override
            public void onClick(DialogInterface dialog, int which) {
                mSlotSelected = (byte) which;
            }
        };

        DialogInterface.OnClickListener OkClick = new DialogInterface.OnClickListener() {
            public void onClick(DialogInterface dialog, int which) {
                open();
            }
        };

        mSlotDialog.setPositiveButton("OK", OkClick);
        mSlotDialog.setTitle("Select Slot Number");
        mSlotDialog.setSingleChoiceItems(arraySlot, 0, Select);
    }


    /**
     * @brief helper class for init smartcard reader
     */
    private class OpenTask extends AsyncTask<Context, Void, Boolean> {
        private byte[] sn = null;
        private int protocol = 0;
        private int errCode;

        @Override
        protected Boolean doInBackground(Context... params) {
            if (isCancelled()) {
                return false;
            }
            try {
				//init ISO7816 mode
                mSlotCount = mSmartCardManager.init(getApplicationContext());
                mSmartCardManager.selectSlot(mSlotSelected);
                Log.d(TAG, "Select slot:" + mSlotSelected);
                mSmartCardManager.powerOn();
                Log.d(TAG, "Power On");
                sn = mSmartCardManager.getSN();
                Log.d(TAG, "get SN");
                protocol = mSmartCardManager.getProtocol();
                Log.d(TAG, "get Protocol");
            } catch (SmartPosException e) {
                mSmartCardManager.deinit();
                errCode = e.getErrorCode();
                return false;
            }
            return true;
        }

        @Override
        protected void onPostExecute(Boolean result) {
            try {
                Message msg = new Message();
                msg.what = MSG_OPEN_RESULT;
                Bundle bundle = new Bundle();
                bundle.putBoolean(KEY_RESULT, result);
                if (result) {
                    String snStr = Utils.bytesToHexString(sn);
                    bundle.putString(KEY_SN, snStr);
                    bundle.putInt(KEY_PROTOCOL, protocol);
                } else {
                    bundle.putInt(KEY_ERR_CODE, errCode);
                }
                msg.setData(bundle);
                mHandler.sendMessage(msg);
            } catch (Exception e) {
                Log.d(TAG, "open task err!");
            }
        }
    }

    /**
     * @brief helper class for deinit smartcard reader
     */
    private class CloseTask extends AsyncTask<Void, Void, Boolean> {
        int errCode;

        @Override
        protected Boolean doInBackground(Void... params) {
            if (isCancelled()) {
                return false;
            }
            try {
                mSmartCardManager.powerOff();
                Log.d(TAG, "Power Off");
                mSmartCardManager.deinit();
                Log.d(TAG, "Deinit");
            } catch (SmartPosException e) {
                mSmartCardManager.deinit();
                errCode = e.getErrorCode();
                return false;
            }
            return true;
        }

        @Override
        protected void onPostExecute(Boolean result) {
            Message msg = new Message();
            msg.what = MSG_CLOSE_RESULT;
            Bundle bundle = new Bundle();
            bundle.putBoolean(KEY_RESULT, result);
            if (!result) {
                bundle.putInt(KEY_ERR_CODE, errCode);
            }
            msg.setData(bundle);
            mHandler.sendMessage(msg);
        }
    }
}
