
package com.paydevice.smartpos.demo.smartcard;

import android.annotation.SuppressLint;
import android.support.v7.app.AppCompatActivity;
import android.app.AlertDialog;
import android.app.AlertDialog.Builder;
import android.app.ProgressDialog;
import android.content.Context;
import android.content.DialogInterface;
import android.os.AsyncTask;
import android.os.Bundle;
import android.os.Handler;
import android.os.Message;
import android.util.Log;
import android.view.View;
import android.view.View.OnClickListener;
import android.widget.Button;
import android.widget.RadioGroup;
import android.widget.TextView;

import com.paydevice.smartpos.demo.R;
import com.paydevice.smartpos.demo.MyToast;
import com.paydevice.smartpos.demo.Utils;
import com.paydevice.smartpos.sdk.SmartPosException;
import com.paydevice.smartpos.sdk.smartcard.SmartCardManager;

public class AT24CActivity extends AppCompatActivity implements RadioGroup.OnCheckedChangeListener {

    private static final String TAG = "AT24CActivity";

    private static final String KEY_RESULT = "result";
    private static final String KEY_ERR_CODE = "code";

	private static final byte CARD_TYPE_SINGLE = 0;
	private static final byte CARD_TYPE_DOUBLE = 1;
	private static final int MAX_WRITE_LENGTH = (128*1024);

    private static final int MSG_OPEN_RESULT = 1;
    private static final int MSG_CLOSE_RESULT = 2;
    private boolean mOpenFlag = false;
    
    private SmartCardManager mSmartCardManager;

	private MyToast mToast;
    private TextView mText;
    private TextView mEditWriteAddr;
    private TextView mEditWriteData;
    private TextView mEditPageSize;
    private TextView mEditReadAddr;
    private TextView mEditReadLen;
	private ProgressDialog mWaitProgress; 

    private Button mOpenBtn;
    private Button mCloseBtn;
    private Button mWriteBtn;
    private Button mReadBtn;
    private OpenTask mOpenTask;
    private CloseTask mCloseTask;

    private Builder mSlotDialog;
    private int mSlotSelected = 0;
    private int mSlotCount = 1;

	private byte mCardType = CARD_TYPE_SINGLE;

    @SuppressLint("HandlerLeak")
    public Handler mHandler = new Handler() {
        public void handleMessage(Message msg) {
            Boolean result = msg.getData().getBoolean(KEY_RESULT);
			if (mWaitProgress != null && mWaitProgress.isShowing())
				mWaitProgress.dismiss();
            switch (msg.what) {
                case MSG_CLOSE_RESULT:
                    if (result) {
                        mToast.showToast(R.string.smartcard_close_success);
                    } else {
						int errCode = msg.getData().getInt(KEY_ERR_CODE);
                        mCloseBtn.setEnabled(false);
						mText.setText("errCode: " + Integer.toHexString(errCode) + " " + SmartCardManager.errCode2String(errCode));
                        mToast.showToast(R.string.smartcard_close_failed);
                    }
                    mOpenBtn.setEnabled(true);
					mWriteBtn.setEnabled(false);
					mReadBtn.setEnabled(false);
                    mOpenFlag = false;
                    break;


                case MSG_OPEN_RESULT:
                    if (result) {
                        mCloseBtn.setEnabled(true);
						mWriteBtn.setEnabled(true);
						mReadBtn.setEnabled(true);
                        mOpenFlag = true;
                        mToast.showToast(R.string.smartcard_open_success);
                    } else {
						int errCode = msg.getData().getInt(KEY_ERR_CODE);
                        if (errCode == SmartCardManager.SCR_ERR_NO_INIT) {
                            init();
                            if (mSlotCount > 1) {
                                mSlotDialog.show();
                            } else {
                                open();
                            }
                        } else {
                            mOpenBtn.setEnabled(true);
							mText.setText("errCode: " + Integer.toHexString(errCode) + " " + SmartCardManager.errCode2String(errCode));
                            mToast.showToast(R.string.smartcard_open_failed);
                        }
                    }
                    break;
            }
        }
    };

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.at24c);
        setTitle(R.string.smartcard_mode_at24c);

        mText = findViewById(R.id.text_result);
        mEditWriteAddr = findViewById(R.id.edit_write_addr);
        mEditPageSize = findViewById(R.id.edit_write_pagesize);
        mEditWriteData = findViewById(R.id.edit_write_data);
        mEditReadAddr = findViewById(R.id.edit_read_addr);
        mEditReadLen = findViewById(R.id.edit_read_length);

        mOpenBtn = findViewById(R.id.btn_open);
        mCloseBtn = findViewById(R.id.btn_close);
        mWriteBtn = findViewById(R.id.btn_write);
        mReadBtn = findViewById(R.id.btn_read);
        mWriteBtn.setEnabled(false);
        mReadBtn.setEnabled(false);
        mCloseBtn.setEnabled(false);

		mWaitProgress =  new ProgressDialog(this);  
		mWaitProgress.setCancelable(false);
		mWaitProgress.setProgressStyle(ProgressDialog.STYLE_SPINNER);

		RadioGroup radioGroup = (RadioGroup) findViewById(R.id.radioGroup);        
		radioGroup.setOnCheckedChangeListener(this);

        setReaderSlotView();

		mSmartCardManager = new SmartCardManager();
		mToast = new MyToast(this);
        //we just init reader to get slot count
        init();
        mSmartCardManager.deinit();
    }

    @Override
    protected void onPause() {
        super.onPause();
		mToast.cancel();
    }

    @Override
    public void onDestroy() {
        //NOTE: power off card and deinit
        Log.d(TAG, "onDestroy()");
        if (mOpenFlag) {
            close();
        } else {
            mSmartCardManager.deinit();
            Log.d(TAG, "Deinit");
            if (mCloseTask != null && mCloseTask.getStatus() != AsyncTask.Status.FINISHED) {
                mCloseTask.cancel(true);
            }
        }
        if (mOpenTask != null && mOpenTask.getStatus() != AsyncTask.Status.FINISHED) {
            mOpenTask.cancel(true);
        }

        super.onDestroy();
    }

	@Override
	public void onCheckedChanged(RadioGroup group, int checkedId) {
		switch (checkedId) {
			case R.id.radio_single:
				Log.d(TAG, "select single byte");
				mCardType = CARD_TYPE_SINGLE;
				break;
			case R.id.radio_double:
				Log.d(TAG, "select double byte");
				mCardType = CARD_TYPE_DOUBLE;
				break;
		}
	}

	public void onOpenBtnClick(View v) {
		if (mSlotCount > 1) {
			mSlotDialog.show();
		} else {
			open();
		}
	}

	public void onCloseBtnClick(View v) {
		mText.setText("");
		mWaitProgress.setMessage("Closing...");
		mWaitProgress.show();
		close();
	}

	public void OnWriteClick(View v) {
		//address
		String tmp = mEditWriteAddr.getText().toString();
		int addr = Integer.parseInt(tmp,16);
		//pagesize
		tmp = mEditPageSize.getText().toString();
		int pagesize = Integer.parseInt(tmp,16);
		//data
		tmp = mEditWriteData.getText().toString();
		byte[] data = Utils.stringToBytes(tmp);
		Log.d(TAG, "write address="+addr+" pageSize="+pagesize+" len:"+data.length);

		int code = mSmartCardManager.AT24CSeriesCmdWrite(addr, mCardType, pagesize, data.length, data); 
		if (code == 0) {
			mText.setText("write success");
		} else {
			mText.setText("write err");
		}
	}

	public void OnReadClick(View v){
		//address
		String tmp = mEditReadAddr.getText().toString();
		int addr = Integer.parseInt(tmp,16);
		//length
		tmp = mEditReadLen.getText().toString();
		int len = Integer.parseInt(tmp,16);
		//data
		byte[] data = new byte[MAX_WRITE_LENGTH];
		int [] dataLen = new int[1];
		Log.d(TAG, "read address="+addr+" len="+ len);
		int code = mSmartCardManager.AT24CSeriesCmdRead(addr, mCardType, len, data, dataLen);
		if (code == 0) {
			Log.d(TAG,"read data length:"+dataLen[0]);
			mText.setText(Utils.bytesToHexString(data));
		} else {
			mText.setText("read err");
		}
	}

    private void init() {
        int errCode;
        String errMsg = null;
        try {
			//init AT24C mode
			mSlotCount = mSmartCardManager.init(getApplicationContext(), SmartCardManager.SCR_MODE_AT24C);
            Log.d(TAG, "got slot count:" + mSlotCount);
        } catch (SmartPosException e) {
            mOpenBtn.setEnabled(false);
            errCode = e.getErrorCode();
            switch (errCode) {
                case SmartCardManager.SCR_ERR_INIT:
                    errMsg = "Reader init error!";
                    break;
            }
            mText.setText("errCode: " + Integer.toHexString(errCode) + "  " + errMsg);
            mToast.showToast(R.string.smartcard_init_err);
        }
    }

    private void open() {
        Log.d(TAG, "open slot:" + mSlotSelected);
        mOpenBtn.setEnabled(false);
        mOpenTask = new OpenTask();
        mOpenTask.execute(this);
    }

    private void close() {
        mCloseBtn.setEnabled(false);
        mCloseTask = new CloseTask();
        mCloseTask.execute();
    }

    private void setReaderSlotView() {
        final String[] arraySlot = new String[]{"slot:0", "Slot:1"};
        mSlotDialog = new AlertDialog.Builder(this);
        DialogInterface.OnClickListener Select = new DialogInterface.OnClickListener() {

            @Override
            public void onClick(DialogInterface dialog, int which) {
                mSlotSelected = (byte) which;
            }
        };

        DialogInterface.OnClickListener OkClick = new DialogInterface.OnClickListener() {
            public void onClick(DialogInterface dialog, int which) {
                open();
            }
        };

        mSlotDialog.setPositiveButton("OK", OkClick);
        mSlotDialog.setTitle("Select Slot Number");
        mSlotDialog.setSingleChoiceItems(arraySlot, 0, Select);
    }


    /**
     * @brief helper class for init smartcard reader
     */
    private class OpenTask extends AsyncTask<Context, Void, Boolean> {
        private int errCode;

		@Override 
		protected void onPreExecute() {
			mWaitProgress.setMessage("Opening...");
			mWaitProgress.show();
		}

        @Override
        protected Boolean doInBackground(Context... params) {
            if (isCancelled()) {
                return false;
            }
            try {
				//init AT24C mode
                mSlotCount = mSmartCardManager.init(getApplicationContext(), SmartCardManager.SCR_MODE_AT24C);
                mSmartCardManager.selectSlot(mSlotSelected);
                Log.d(TAG, "Select slot:" + mSlotSelected);
                mSmartCardManager.powerOn();
                Log.d(TAG, "Power On");
            } catch (SmartPosException e) {
                mSmartCardManager.deinit();
                errCode = e.getErrorCode();
                return false;
            }
            return true;
        }

        @Override
        protected void onPostExecute(Boolean result) {
            try {
                Message msg = new Message();
                msg.what = MSG_OPEN_RESULT;
                Bundle bundle = new Bundle();
                bundle.putBoolean(KEY_RESULT, result);
                if (result) {
                } else {
                    bundle.putInt(KEY_ERR_CODE, errCode);
                }
                msg.setData(bundle);
                mHandler.sendMessage(msg);
            } catch (Exception e) {
                Log.d(TAG, "open task err!");
            }
        }
    }

    /**
     * @brief helper class for deinit smartcard reader
     */
    private class CloseTask extends AsyncTask<Void, Void, Boolean> {
        int errCode;

        @Override
        protected Boolean doInBackground(Void... params) {
            if (isCancelled()) {
                return false;
            }
            try {
                mSmartCardManager.powerOff();
                Log.d(TAG, "Power Off");
                mSmartCardManager.deinit();
                Log.d(TAG, "Deinit");
            } catch (SmartPosException e) {
                mSmartCardManager.deinit();
                errCode = e.getErrorCode();
                return false;
            }
            return true;
        }

        @Override
        protected void onPostExecute(Boolean result) {
            Message msg = new Message();
            msg.what = MSG_CLOSE_RESULT;
            Bundle bundle = new Bundle();
            bundle.putBoolean(KEY_RESULT, result);
            if (!result) {
                bundle.putInt(KEY_ERR_CODE, errCode);
            }
            msg.setData(bundle);
            mHandler.sendMessage(msg);
        }
    }
}
