package com.paydevice.smartpos.demo;

import android.os.Build;
import android.support.v7.app.AppCompatActivity;
import android.app.PendingIntent;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.IntentFilter.MalformedMimeTypeException;
import android.nfc.FormatException;
import android.nfc.NdefRecord;
import android.nfc.NdefMessage;
import android.nfc.NfcAdapter;
import android.nfc.Tag;
import android.nfc.tech.IsoDep;
import android.nfc.tech.MifareClassic;
import android.nfc.tech.MifareUltralight;
import android.nfc.tech.Ndef;
import android.nfc.tech.NfcA;
import android.nfc.tech.NfcB;
import android.nfc.tech.NfcBarcode;
import android.nfc.tech.NfcF;
import android.nfc.tech.NfcV;
import android.nfc.tech.NdefFormatable;
import android.os.Bundle;
import android.util.Log;
import android.view.View;
import android.view.View.OnClickListener;
import android.widget.Button;
import android.widget.TextView;

import java.io.IOException;
import java.nio.charset.Charset;
import java.lang.System;

public class NfcActivity extends AppCompatActivity {

    private static final String TAG = "NfcActivity";

	private MyToast mToast;
    private TextView mTagText;
    private TextView mEditText;

    private NfcAdapter mNfcAdapter;
    private PendingIntent mPendingIntent;
    private IntentFilter[] mFilters;
    private String[][] mTechLists;
    private Tag mTag;


    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_nfc);
        setTitle(R.string.nfc);

		mToast = new MyToast(this);
        mTagText = (TextView) this.findViewById(R.id.text);
        mEditText = (TextView) this.findViewById(R.id.edit);

        Button mWriteBtn = (Button) this.findViewById(R.id.write_btn);
        mWriteBtn.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View arg) {
                writeNdefTag(mTag, mEditText.getText().toString());
            }
        });
        Button mReadBtn = (Button) this.findViewById(R.id.read_btn);
        mReadBtn.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View arg) {
                readNdefTag(mTag);
            }
        });


        //prepare NFC
        mNfcAdapter = NfcAdapter.getDefaultAdapter(this);
        if (mNfcAdapter == null) {
            mToast.showToast(R.string.nfc_not_support);
        } else {
            if (!mNfcAdapter.isEnabled()) {
                mToast.showToast(R.string.nfc_disabled);
            } else {
                // Create a generic PendingIntent that will be deliver to this activity. The NFC stack
                // will fill in the intent with the details of the discovered tag before delivering to
                // this activity.
                int flag = 0;
                //Android12+ required
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S) {
                    flag = PendingIntent.FLAG_MUTABLE;
                }
                mPendingIntent = PendingIntent.getActivity(this, 0,
                        new Intent(this, getClass()).addFlags(Intent.FLAG_ACTIVITY_SINGLE_TOP), flag);

                // Setup an intent filter for all MIME based dispatches
                IntentFilter ndef = new IntentFilter(NfcAdapter.ACTION_NDEF_DISCOVERED);
                try {
                    ndef.addDataType("*/*");
                } catch (MalformedMimeTypeException e) {
                    throw new RuntimeException("fail", e);
                }
                mFilters = new IntentFilter[]{
                        ndef,
                };

                // Setup a tech list for all Ndef tags
                mTechLists = new String[][]{
                        new String[]{IsoDep.class.getName()},
                        new String[]{MifareClassic.class.getName()},
                        new String[]{MifareUltralight.class.getName()},
                        new String[]{Ndef.class.getName()},
                        new String[]{NfcA.class.getName()},
                        new String[]{NfcB.class.getName()},
                        new String[]{NfcBarcode.class.getName()},
                        new String[]{NfcF.class.getName()},
                        new String[]{NfcV.class.getName()},
                };
            }
        }
    }

    @Override
    protected void onResume() {
        super.onResume();

        if (mNfcAdapter != null && mPendingIntent != null) {
            mNfcAdapter.enableForegroundDispatch(this, mPendingIntent, mFilters, mTechLists);
        }
    }

    @Override
    public void onPause() {
        super.onPause();
		mToast.cancel();

        if (mNfcAdapter != null) {
            mNfcAdapter.disableForegroundDispatch(this);
        }
    }

    @Override
	public void onNewIntent(Intent intent) {
		Log.i(TAG, "Discovered tag with intent: " + intent);
		mTag = intent.getParcelableExtra(NfcAdapter.EXTRA_TAG);
		String[] techList = mTag.getTechList();
		Log.d(TAG,"======tag tech list======");
		for(int i=0;i<techList.length;i++) {
			Log.d(TAG,techList[i]);
			if (techList[i].equals(MifareClassic.class.getName())) {
				readMifareClassic(mTag);
				break;
			} else if (techList[i].equals(Ndef.class.getName())) {
				readNdefTag(mTag);
				break;
			} else if (techList[i].equals(IsoDep.class.getName())) {
				readIsoDep(mTag);
				break;
			} else if (techList[i].equals(NfcA.class.getName())) {
				readNfcA(mTag);
				//readNdefTag(mTag);
				break;
			} else if (techList[i].equals(NfcB.class.getName())) {
				readNfcB(mTag);
				break;
			} else {
				Log.d(TAG,"TODO: need parser!");
				break;
			}
		}
	}

    //refer: http://nfc-tools.org/index.php?title=ISO14443A
    private void readNfcA(Tag tag) {
        NfcA nfca = NfcA.get(tag);
        try {
            byte[] id = nfca.getTag().getId();
            byte[] atqa = nfca.getAtqa();
            short sak = nfca.getSak();
            int maxSize = nfca.getMaxTransceiveLength();
            Log.d(TAG, "id:" + Utils.bytesToHexString(id));
            Log.d(TAG, "ATQA:" + Utils.bytesToHexString(atqa));
            Log.d(TAG, "SAK:" + sak);
            Log.d(TAG, "maxSize:" + maxSize);
			mTagText.setText("UID:"+Utils.bytesToHexString(id));
            //ISO/IEC 14443-3 Mifare Ultralight
            if (atqa[0] == (byte) 0x44 && atqa[1] == (byte) 0x0) {
                mToast.showToast("Type: MIFARE Ultralight");
            } else {
                /* TODO:other format */
            }
            /* TODO: read page?? */
            nfca.connect();
			/*
			byte[] respone = nfca.transceive(cmdRead);
			respone = nfca.transceive(cmdRead);
			Log.d(TAG,"cmdRead  return:"+Utils.bytesToHexString(respone));
			*/
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            try {
                nfca.close();
            } catch (IOException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            }
        }
    }

	private void readNfcB(Tag tag) {
		NfcB nfcb = NfcB.get(tag);
		try {
			byte[] id = nfcb.getTag().getId();
			byte[] appData = nfcb.getApplicationData();
			byte[] protocalData = nfcb.getProtocolInfo();
			int maxSize = nfcb.getMaxTransceiveLength();
			Log.d(TAG,"id:"+Utils.bytesToHexString(id));
			Log.d(TAG,"appData:"+Utils.bytesToHexString(appData));
			Log.d(TAG,"protocalData:"+Utils.bytesToHexString(protocalData));
			Log.d(TAG,"maxSize:"+maxSize);

			nfcb.connect();
			//China IDcard antenna freq not a standard 13.56MHz, recommend read card 2~3times to avoid read failed
			byte[] query = {(byte)0x05,(byte)0x00,(byte)0x00};
			byte[] select = {(byte)0x1D,(byte)0x00,(byte)0x00,(byte)0x00,(byte)0x00,(byte)0x00,(byte)0x08,(byte)0x01,(byte)0x08};
			byte[] guid = {(byte)0x00,(byte)0x36,(byte)0x00,(byte)0x00,(byte)0x08};
			byte[] respone = nfcb.transceive(query);
			//Log.d(TAG,"query return("+respone.length+"):"+Utils.bytesToHexString(respone));
			respone = nfcb.transceive(select);
			//Log.d(TAG,"select return("+respone.length+"):"+Utils.bytesToHexString(respone));
			respone = nfcb.transceive(guid);
			//Log.d(TAG,"guid return("+respone.length+"):"+Utils.bytesToHexString(respone));
			mTagText.setText("UID:"+Utils.bytesToHexString(respone).substring(0,25));
		} catch (IOException e) {
			e.printStackTrace();
		} finally {
			try {
				nfcb.close();
			} catch (IOException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		}
	}

    private void readIsoDep(Tag tag) {
        IsoDep iso = IsoDep.get(tag);
		byte[] id = iso.getTag().getId();
		mTagText.setText("UID:"+Utils.bytesToHexString(id));
        //debug,refer china PBOC3.0
        byte[] cmdReset = {(byte) 0x00, (byte) 0xA4, (byte) 0x04, (byte) 0x00, (byte) 0x08, (byte) 0xA0, (byte) 0x00, (byte) 0x00, (byte) 0x03, (byte) 0x33, (byte) 0x01, (byte) 0x01, (byte) 0x01};
        byte[] cmdRead = {(byte) 0x00, (byte) 0xB2, (byte) 0x01, (byte) 0x14, (byte) 0x00};
        try {
            iso.connect();
            byte[] respone = iso.transceive(cmdReset);
            Log.d(TAG,"cmdReset return:"+Utils.bytesToHexString(respone));
            respone = iso.transceive(cmdRead);
            Log.d(TAG,"cmdRead  return:" + Utils.bytesToHexString(respone));
            int i;
            for (i = 0; i < respone.length; i++) {
                if (respone[i] == 0x5A) {
                    int accountLen = respone[i + 1];
                    byte[] accountBytes = new byte[accountLen];
                    System.arraycopy((Object)respone, i + 2, (Object)accountBytes, 0, accountLen);
                    String accountStr = Utils.bytesToHexString(accountBytes);
                    mTagText.setText(getString(R.string.nfc_card_account) + accountStr);
                    break;
                }
            }
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            try {
                iso.close();
            } catch (IOException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            }
        }
    }

    private void readMifareClassic(Tag tag) {
        MifareClassic mc = MifareClassic.get(tag);
        try {
            boolean auth = false;
            StringBuilder metaInfo = new StringBuilder();
            mc.connect();
            int type = mc.getType();
            int sectorCount = mc.getSectorCount();
            String typeS = "";
            switch (type) {
                case MifareClassic.TYPE_CLASSIC:
                    typeS = "TYPE_CLASSIC";
                    break;
                case MifareClassic.TYPE_PLUS:
                    typeS = "TYPE_PLUS";
                    break;
                case MifareClassic.TYPE_PRO:
                    typeS = "TYPE_PRO";
                    break;
                case MifareClassic.TYPE_UNKNOWN:
                    typeS = "TYPE_UNKNOWN";
                    break;
            }
            metaInfo.append("card type: ").append(typeS).append(" sectors: ").append(sectorCount).append(" blocks: ").append(mc.getBlockCount()).append(" size: ").append(mc.getSize()).append("byte\n");
            mTagText.setText(metaInfo.toString());
            for (int j = 0; j < sectorCount; j++) {
                //Authenticate a sector with key A.
                auth = mc.authenticateSectorWithKeyA(j, MifareClassic.KEY_NFC_FORUM);
                int bCount;
                int bIndex;
                if (auth) {
                    metaInfo.append("Sector ").append(j).append(":auth ok\n");
                    bCount = mc.getBlockCountInSector(j);
                    bIndex = mc.sectorToBlock(j);
                    for (int i = 0; i < bCount; i++) {
                        byte[] data = mc.readBlock(bIndex);
                        metaInfo.append("Block ").append(bIndex).append(" : ").append(Utils.bytesToHexString(data)).append("\n");
                        bIndex++;
                    }
                } else {
                    metaInfo.append("Sector ").append(j).append(":auth fail\n");
                }
            }
            Log.d(TAG, metaInfo.toString());
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            try {
                mc.close();
            } catch (IOException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            }
        }
    }

    private void readNdefTag(Tag tag) {
        if (tag != null) {
            Ndef ndef = Ndef.get(tag);
            if (ndef != null) {
                try {
                    ndef.connect();
                    NdefMessage msg = ndef.getNdefMessage();
                    if (msg != null) {
                        NdefRecord record = msg.getRecords()[0];
                        if (record != null) {
                            String readResult = new String(record.getPayload(), "UTF-8");
                            Log.d(TAG, "payload:" + record.toString());
                            mTagText.setText("payload:" + readResult);
                        }
                    } else {
                        Log.d(TAG, "tag is empty!");
                    }
                } catch (IOException ie) {
                    ie.printStackTrace();
                } catch (FormatException fe) {
                    fe.printStackTrace();
                } finally {
                    try {
                        ndef.close();
                    } catch (IOException e) {
                        e.printStackTrace();
                    }
                }
            } else {
                mToast.showToast("Tag does not support Ndef");
            }
        }
    }

    private void writeNdefTag(Tag tag, String text) {
        if (tag != null) {
            Ndef ndef = Ndef.get(tag);
            try {
                NdefRecord record = createTextRecord(text);
                NdefMessage msg = new NdefMessage(record);
                if (ndef != null) {
                    ndef.connect();
                    if (!ndef.isWritable()) {
                        mTagText.setText("Tag was readonly!");
                        return;
                    }
                    if (ndef.getMaxSize() < msg.toByteArray().length) {
                        mTagText.setText("Tag is full!");
                        return;
                    }
                    ndef.writeNdefMessage(msg);
                    mTagText.setText("write success!");
                } else {
                    NdefFormatable format = NdefFormatable.get(mTag);
                    if (format != null) {
                        format.connect();
                        format.format(msg);
                        mTagText.setText("write success!");
                    } else {
                        mTagText.setText("Tag doesn't support NDEF format!");
                    }
                }
            } catch (IOException ie) {
                ie.printStackTrace();
            } catch (FormatException fe) {
                fe.printStackTrace();
            } finally {
                try {
                    if (ndef != null) {
                        ndef.close();
                    }
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
    }

    private NdefRecord createTextRecord(String text) {
        byte[] textBytes = text.getBytes(Charset.forName("UTF-8"));
        return new NdefRecord(NdefRecord.TNF_WELL_KNOWN, NdefRecord.RTD_TEXT, new byte[0], textBytes);
    }

}
