package com.paydevice.smartpos.demo;

import android.content.Intent;
import android.content.pm.ActivityInfo;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.os.AsyncTask;
import android.os.Bundle;
import android.os.Handler;
import android.os.Message;
import android.support.v7.app.AppCompatActivity;
import android.support.v7.widget.Toolbar;
import android.util.DisplayMetrics;
import android.util.Log;
import android.view.Menu;
import android.view.MenuItem;
import android.view.View;
import android.widget.Button;

import com.paydevice.smartpos.demo.printer.PosSalesSlip;

import com.paydevice.smartpos.sdk.SmartPosException;
import com.paydevice.smartpos.sdk.printer.PrinterManager;
import com.paydevice.smartpos.sdk.printer.Printer;
import com.paydevice.smartpos.sdk.printer.UsbPrinter;
import com.paydevice.smartpos.sdk.printer.SerialPortPrinter;
import com.paydevice.smartpos.sdk.cashdrawer.CashDrawer;

public class MainActivity extends AppCompatActivity {

    private static final String TAG = "SmartPosDemo";

    private static final boolean havePrinter = true;
    private static final boolean haveScr = true;
    private static final boolean haveSecondary = true;

    private static final boolean haveCamera = true;
    private static final boolean haveNfc = true;
    private static final boolean haveSerailport = true;

    private static final boolean haveCashbox = true;
    private static final boolean haveMcr = true;
    private static final boolean haveFp = true;
    private static final boolean haveSam = true;


    private static final String KEY_PRINTING_RESULT = "printing_result";
    private static final String KEY_PRINTER_UPDATE = "printer_update_result";

    //Message type
    private static final int MSG_PRINTING_RESULT = 1;
    private static final int MSG_PRINTER_UPDATE_START = 2;
    private static final int MSG_PRINTER_UPDATE_DONE = 3;
    private static final int MSG_PRINTER_SET_BMP_NVRAM = 4;
    private static final int MSG_PRINTER_DEL_BMP_NVRAM = 5;

    //UI
	private MyToast mToast;
    private Button printerBtn;
    private Button scrBtn;
    private Button secondaryBtn;
    private Button cameraBtn;
    private Button scannerBtn;
    private Button nfcBtn;
    private Button serialportBtn;
    private Button cashdrawerBtn;
    private Button mcrBtn;
    private Button fpBtn;
    private Button samBtn;
    private Button ledBtn;

	private Printer mPrinter = null;
	private PrinterManager mPrinterManager = null;

    //for printer
    private PrintTask mPrintTask = null;
    private PosSalesSlip mTemplate = null;

    public Handler mHandler = new Handler() {
        public void handleMessage(Message msg) {
            Boolean result = false;
            switch (msg.what) {
                case MSG_PRINTING_RESULT:
                    result = msg.getData().getBoolean(KEY_PRINTING_RESULT);
                    if (mTemplate != null) {
                        mTemplate.destroy();
                        mTemplate = null;
                    }
                    if (result) {
                        mToast.showToast(R.string.printer_printing_success);
                    } else {
                        mToast.showToast(R.string.printer_printing_failed);
                    }
                    printerBtn.setEnabled(true);
                    break;

                case MSG_PRINTER_UPDATE_START:
                    mToast.showToast("fw update take about 12s, don't interrupt!");
                    printerBtn.setEnabled(false);
                    new Thread("fwupdate") {
                        public void run() {
                            if (mPrinterManager != null) {
                                Bundle bundle = new Bundle();
                                bundle.putBoolean(KEY_PRINTER_UPDATE, true);
								//update fw for serial printer
                                try {
                                    mPrinterManager.fwUpdate(getAssets(), "fw.bin");
                                } catch (SmartPosException e) {
                                    Log.d(TAG, "fw update err:" + e.getErrorCode());
                                    bundle.putBoolean(KEY_PRINTER_UPDATE, false);
                                }

                                Message msg = new Message();
                                msg.setData(bundle);
                                msg.what = MSG_PRINTER_UPDATE_DONE;
                                mHandler.sendMessage(msg);
                            }
                        }
                    }.start();
                    break;

                case MSG_PRINTER_UPDATE_DONE:
                    result = msg.getData().getBoolean(KEY_PRINTER_UPDATE);
                    if (result) {
                        mToast.showToast("fw update done!");
                    } else {
                        mToast.showToast("fw update fail!");
                    }
                    printerBtn.setEnabled(true);
                    break;

                case MSG_PRINTER_SET_BMP_NVRAM:
                    printerBtn.setEnabled(false);
                    result = saveBitmaptoNVRAM();
                    if (result) {
                        mToast.showToast("save bitmaps done!");
                    } else {
                        mToast.showToast("save bitmaps fail!");
                    }
                    printerBtn.setEnabled(true);
                    break;

                case MSG_PRINTER_DEL_BMP_NVRAM:
                    printerBtn.setEnabled(false);
                    result = deleteBitmapFromNVRAM();
                    if (result) {
                        mToast.showToast("delete bitmaps done!");
                    } else {
                        mToast.showToast("delete bitmaps fail!");
                    }
                    printerBtn.setEnabled(true);
                    break;

                default:
                    break;
            }
        }
    };

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_main);
        Toolbar toolbar = (Toolbar) findViewById(R.id.toolbar);
        setSupportActionBar(toolbar);

        //initialize UI
        initComponent();
    }

    @Override
    protected void onResume() {
        super.onResume();
        //initialize devices
        initDevice();
    }

    @Override
    protected void onPause() {
        super.onPause();
		mToast.cancel();
	}

    @Override
    public boolean onCreateOptionsMenu(Menu menu) {
        // Inflate the menu; this adds items to the action bar if it is present.
        getMenuInflater().inflate(R.menu.menu_main, menu);
        return true;
    }

    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
        // Handle action bar item clicks here. The action bar will
        // automatically handle clicks on the Home/Up button, so long
        // as you specify a parent activity in AndroidManifest.xml.
        int id = item.getItemId();

        //noinspection SimplifiableIfStatement
        Message msg = new Message();
        switch (id) {
            case R.id.action_settings:
                msg.what = MSG_PRINTER_UPDATE_START;
                mHandler.sendMessage(msg);
                break;

            case R.id.action_savetonvram:
				if (Utils.getNvramFlag(this)) {
					mToast.showToast("alreayd saved!");
				} else {
					msg.what = MSG_PRINTER_SET_BMP_NVRAM;
					mHandler.sendMessage(msg);
				}
                break;

            case R.id.action_deletefromnvram:
                msg.what = MSG_PRINTER_DEL_BMP_NVRAM;
                mHandler.sendMessage(msg);
                break;

            case R.id.action_exit:
				finish();
                break;

            default:
                break;
        }

        return super.onOptionsItemSelected(item);
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
        cleanPrinterTask();
    }

	/**
	* @brief initialize devices
	*
	* @return 
	*/
    private void initDevice() {
		//check printer for different models
		//Built-in serialport printer: FH070H-A,FH100H-A,FH070A2,FH101A1-D
		//Built-in usb printer:        FH116A3

		if (mPrinter == null) {
			//if usb printer no found then try serialport printer 
			try {
				//80mm USB printer
				mPrinter = new UsbPrinter(this);
				mPrinter.selectBuiltInPrinter();
				mPrinter.open();
				mPrinter.close();
			} catch (SmartPosException e) {
				Log.d(TAG,"no usb printer,try serialport printer");
				//58mm serialport printer
				mPrinter = new SerialPortPrinter();
				mPrinter.selectBuiltInPrinter();
			}

			mPrinterManager = new PrinterManager(mPrinter,
					(mPrinter.getType() == PrinterManager.PRINTER_TYPE_USB)
					? PrinterManager.TYPE_PAPER_WIDTH_80MM
					: PrinterManager.TYPE_PAPER_WIDTH_58MM);
		}
    }

    /**
     * @return
     * @brief prepare all components
     */
    private void initComponent() {
        //Printer
        printerBtn = (Button) this.findViewById(R.id.print_btn);
        printerBtn.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View arg) {
                testPrinter();
            }
        });
        //SmartCard
        scrBtn = (Button) this.findViewById(R.id.scr_btn);
        scrBtn.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View arg) {
                Intent intent = new Intent(MainActivity.this, SmartCardActivity.class);
                startActivity(intent);
            }
        });
        if (!haveScr) {
            scrBtn.setVisibility(View.GONE);
        }
        //Secondary Display
        secondaryBtn = (Button) this.findViewById(R.id.secondary_btn);
        secondaryBtn.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View arg) {
                Intent intent = new Intent(MainActivity.this, SecondaryDisplayActivity.class);
                startActivity(intent);
            }
        });
        if (!haveSecondary) {
            secondaryBtn.setVisibility(View.GONE);
        }
        //Camera
        cameraBtn = (Button) this.findViewById(R.id.camera_btn);
        cameraBtn.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View arg) {
				DisplayMetrics outMetrics = new DisplayMetrics();
				getWindowManager().getDefaultDisplay().getRealMetrics(outMetrics);
				int widthPixels = outMetrics.widthPixels;
				int heightPixels = outMetrics.heightPixels;
				//Log.i(TAG, "widthPixels = " + widthPixels + ",heightPixels = " + heightPixels);

                Intent intent = new Intent(MainActivity.this, CameraActivity.class);
				if (widthPixels < heightPixels) {
					intent.putExtra("orientation", "portrait");
				} else {
					intent.putExtra("orientation", "landspace");
				}
                startActivity(intent);
            }
        });
		//HID scanner
        scannerBtn = (Button) this.findViewById(R.id.scanner_btn);
        scannerBtn.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View arg) {
                Intent intent = new Intent(MainActivity.this, ScannerActivity.class);
                startActivity(intent);
            }
        });
        //NFC
        nfcBtn = (Button) this.findViewById(R.id.nfc_btn);
        nfcBtn.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View arg) {
                Intent intent = new Intent(MainActivity.this, NfcActivity.class);
                startActivity(intent);
            }
        });
        if (!haveNfc) {
            nfcBtn.setVisibility(View.GONE);
        }
		//Serial port
        serialportBtn = (Button) this.findViewById(R.id.serialport_btn);
        serialportBtn.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View arg) {
                Intent intent = new Intent(MainActivity.this, SerialPortActivity.class);
                startActivity(intent);
            }
        });
        if (!haveSerailport) {
            serialportBtn.setVisibility(View.GONE);
        }
		//Cash drawer
        cashdrawerBtn = (Button) this.findViewById(R.id.cashdrawer_btn);
        cashdrawerBtn.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View arg) {
                openCashDrawer();
                openCashDrawerEx();
            }
        });
        if (!haveCashbox) {
            cashdrawerBtn.setVisibility(View.GONE);
        }
        //MagneticCard
        mcrBtn = (Button) this.findViewById(R.id.mcr_btn);
        mcrBtn.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View arg) {
                Intent intent = new Intent(MainActivity.this, McrActivity.class);
                startActivity(intent);
            }
        });
        if (!haveMcr) {
            mcrBtn.setVisibility(View.GONE);
        }
        //FingerPrint
        fpBtn = (Button) this.findViewById(R.id.fp_btn);
        fpBtn.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View arg) {
                Intent intent = new Intent(MainActivity.this, FingerPrintActivity.class);
                startActivity(intent);
            }
        });
        if (!haveFp) {
            fpBtn.setVisibility(View.GONE);
        }
        //SAM Card
        samBtn = (Button) this.findViewById(R.id.sam_btn);
        samBtn.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View arg) {
                Intent intent = new Intent(MainActivity.this, SamCardActivity.class);
                startActivity(intent);
            }
        });
        if (!haveSam) {
            samBtn.setVisibility(View.GONE);
        }
        //LED
        ledBtn = (Button) this.findViewById(R.id.led_btn);
        ledBtn.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View arg) {
                Intent intent = new Intent(MainActivity.this, LedActivity.class);
                startActivity(intent);
            }
        });

		mToast = new MyToast(this);
    }

	/**
	* @brief open cashdrawer with built-in RJ11. FH070H-A,FH100H-A no RJ11 port
	*        pls check your device model
	*
	* @return 
	*/
    private void openCashDrawer() {
		CashDrawer.open();
    }

	/**
	* @brief open cashdrawer via USB-RJ11 converter
    *        only support PayDevice USB-RJ11 converter(serialport node is /dev/ttyCB0,dev/ttyCB1,etc.)
	*
	* @return 
	*/
	private void openCashDrawerEx() {
		CashDrawer.openEx();
	}

    /*
     * Total size of NVRAM is 64KB.Bitmap width and height must be an integer multiple of 8.
     * Warning: You should not write NVRAM often, it maybe cause NVRAM broken.
     * Recommended less than 10 times a day.
     * You should save bitmaps only once and then just print it every times.
     */
    private boolean saveBitmaptoNVRAM() {
        boolean ret = true;
		//NOTE:avoid bitmap scaled by lcd density
		BitmapFactory.Options options = new BitmapFactory.Options();
		options.inTargetDensity = 0;
		options.inScaled = false;//disable scaled, must be original size
		Bitmap logo = BitmapFactory.decodeResource(getResources(), R.drawable.logo, options);
		Bitmap logo2 = BitmapFactory.decodeResource(getResources(), R.drawable.logo_large, options);
		Bitmap[] bmpArray = new Bitmap[2];
		bmpArray[0] = logo;
		bmpArray[1] = logo2;
		Log.d(TAG,"logo1 wxh:"+logo.getWidth()+"x"+logo.getHeight());
		Log.d(TAG,"logo2 wxh:"+logo2.getWidth()+"x"+logo2.getHeight());
        try {
            mPrinterManager.connect();
            mPrinterManager.cmdSaveBitmapToNVRAM(bmpArray);
        } catch (SmartPosException e) {
            Log.d(TAG, "save bitmap to NVRAM err:" + e.getErrorCode());
            ret = false;
        } finally {
            try {
                Thread.sleep(500);
            } catch (InterruptedException ie) {
            }
			try {
				mPrinterManager.disconnect();
			} catch (SmartPosException e) {
			}
			logo.recycle();
			logo2.recycle();
			logo = logo2 = null;
			Utils.setNvramFlag(this, true);
        }
        return ret;
    }

	/**
	 * @brief delete all bitmaps from NVRAM
	 *
	 * @return 
	 */
    private boolean deleteBitmapFromNVRAM() {
        boolean ret = true;
        try {
            mPrinterManager.connect();
            mPrinterManager.cmdDeleteBitmapFromNVRAM();
        } catch (SmartPosException e) {
            Log.d(TAG, "delete bitmaps from NVRAM err:" + e.getErrorCode());
            ret = false;
        } finally {
			Utils.setNvramFlag(this, false);
            try {
                Thread.sleep(500);
            } catch (InterruptedException ie) {
            }
			try {
				mPrinterManager.disconnect();
			} catch (SmartPosException e) {
			}
        }
        return ret;
	}

    /**
     * @return
     * @brief test printing
     */
    private void testPrinter() {
        /* prepare a print template */
        if (mTemplate == null) {
            mTemplate = new PosSalesSlip(this, mPrinterManager);
        }
        int err = mTemplate.prepare();
        if (err == 0) {
            //fill receipt template
            mTemplate.setMerchantName("CocoPark");
            mTemplate.setMerchantNo("102510154110045");
            mTemplate.setTerminalNo("10095066");
            mTemplate.setOperatorNo("01");
            mTemplate.setIssuer("03050001");
            mTemplate.setCardNo("622602******9376");
            mTemplate.setTxnType("SALE");
            mTemplate.setBatchNo("000814");
            mTemplate.setVoucherNo("003707");
            mTemplate.setAuthNo("381936");
            mTemplate.setExpDate("0000");
            mTemplate.setRefNo("103758494052");
            mTemplate.setDate("2012-11-25 10:37:58");
            mTemplate.setAmount("67.10");
            //start AsyncTask
            printerBtn.setEnabled(false);
            mPrintTask = new PrintTask();
            mPrintTask.executeOnExecutor(AsyncTask.THREAD_POOL_EXECUTOR, mTemplate);
        } else {
            Log.d(TAG, "printer prepare err code: " + err);
            switch (err) {
                case PrinterManager.PRINTER_ERR_OPEN:
                    mToast.showToast(R.string.printer_init_err);
                    break;
                case PrinterManager.PRINTER_ERR_NO_PAPER:
                    mToast.showToast(R.string.printer_no_paper);
                    break;
                case PrinterManager.PRINTER_ERR_PARAM:
                    Log.d(TAG, "param err!");
                    break;

                default:
                    break;
            }
            mTemplate.destroy();
            mTemplate = null;
        }
    }

    private void cleanPrinterTask() {
        Log.d(TAG, "---cleanPrinterTask---");
        if (mTemplate != null) {
            mTemplate.destroy();
            mTemplate = null;
        }
        if (mPrintTask != null && mPrintTask.getStatus() != AsyncTask.Status.FINISHED) {
            mPrintTask.cancel(true);
			mPrintTask = null;
        }
    }

    /**
     * @brief Printing task class
     */
    private class PrintTask extends AsyncTask<PosSalesSlip, Void, Boolean> {
        protected Boolean doInBackground(PosSalesSlip... templates) {
			Thread.currentThread().setPriority(Thread.MAX_PRIORITY);
            if (isCancelled()) {
                return false;
            }
            try {
                if (templates[0].validate()) {
                    templates[0].print();
                } else {
                    Log.d(TAG, "mTemplate data illegal!");
                    return false;
                }
            } catch (SmartPosException e) {
                return false;
            }
            return true;
        }

        protected void onPostExecute(Boolean result) {
            Bundle bundle = new Bundle();
            bundle.putBoolean(KEY_PRINTING_RESULT, result);
            Message msg = new Message();
            msg.what = MSG_PRINTING_RESULT;
            msg.setData(bundle);
            mHandler.sendMessage(msg);
        }
    }
}
